import numpy as np
import scipy.signal as sig

def create_time_data(num, samp_freq):
    
    delta_t = 1./samp_freq
    start, stop = 0, (num-1)*delta_t

    t = np.linspace(start, stop, num)

    return t


def create_waveform_data(t, a, f, shape):

    num = len(t)

    if shape == 'Sine':
        y = a*np.sin(2*np.pi*f*t)
    elif shape == 'Cosine':
        y = a*np.cos(2*np.pi*f*t)
    elif shape == 'DC Level':
        y = a*np.ones(num)
    elif shape == 'Square':
        y = a*sig.square(2*np.pi*f*t)
    elif shape == 'Sawtooth':
        y = a*sig.sawtooth(2*np.pi*f*t, 1.0)
    elif shape == 'Triangle':
        y = a*sig.sawtooth(2*np.pi*f*t, 0.5)
    elif shape == 'User-Defined':
        y = (5*np.sin(2*np.pi*50*t)
             + 5*np.sin(2*np.pi*100*t))

    return y


def sum_sines(t, num_terms):

    num = len(t)
    y = np.zeros(num)
    largest_odd = 2*num_terms + 1
    

    for odd in range(1, largest_odd, 2):
       y += (1/odd)*np.sin(2*np.pi*odd*t)
    y = (4/np.pi)*y

    return y


def projectile(thetas, v, h):

    g = 9.8
    ranges = []
    for theta in thetas:

        theta_rad = theta*np.pi/180
        x = np.sin(2*theta_rad)
        y = np.cos(theta_rad)

        alpha = v**2/(2*g)
        beta = 8*g*h/v**2
   
        r = alpha*(x + np.sqrt(x**2 + beta*y**2))
        ranges.append(r)

        max_index = np.argmax(ranges)
        max_angle = thetas[max_index]

    return ranges, max_angle


def falling_in_air(num,delta_t,alpha):\
    
    v = np.zeros(num)
    g = 9.8
    for i in range(2,num):
        v[i] = v[i-1] + (g-alpha*v[i-1]**2)*delta_t

    return v



if __name__ == '__main__':

    import matplotlib.pyplot as plt

    num, samp_freq = 100, 100
    a, f = 5, 3
    shape = 'Sine'
    
    t = create_time_data(num, samp_freq)
    y = create_waveform_data(t, a, f, shape)
    
    plt.plot(t, y)
    plt.show()
